// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "hw_gx.h"
#include "messages.h"
#include "gp.h"
#include "registers.h"
#include "common.h"

#define DEFINE_W_FIFO(gem, type) \
	void HwGX::w##gem##_fifo(Hardware *h, WORD offset, type data) {\
	w_fifo<type>(h, offset, data); }
DEFINE_W_FIFO(b, BYTE)
DEFINE_W_FIFO(h, WORD)
DEFINE_W_FIFO(w, DWORD)
DEFINE_W_FIFO(d, QWORD)
template<class T> void HwGX::w_fifo(Hardware *h, WORD offset, T data) {
	ADDITIVE_TIMING(&h->m_hwt[HWT_FIFO]);
	MYASSERT(offset == 0x8000);
	FIFODEGUB("FIFO write: 0x");
#pragma warning(push)
#pragma warning(disable:4127)
	if(sizeof(T) == 8) {
#pragma warning(pop)
		FIFODEGUB("%016I64X\n", data);
	} else {
		FIFODEGUB("%0*X\n", sizeof(data)*2, data);
	}
	BYTE *old_pWrite = h->fifo.pTempWriteptr;
	T swapped = tswap(data);
	if(h->fifo.pTempWriteptr + sizeof(data) >= h->fifo.pEnd) {
		wrapWrite(h, (BYTE *)&swapped, sizeof(data));
	} else {
		MAKE(T, *h->fifo.pTempWriteptr) = swapped;
		h->fifo.pTempWriteptr += sizeof(data);
	}
	FIFODEGUB("Temp Writeptr: 0x%08X + 0x%X = 0x%08X\n",
		PTR2ADDRESS(old_pWrite), sizeof(data), PTR2ADDRESS(h->fifo.pTempWriteptr));
	if(h->fifo.linked) {
		/*if(g::use_map) {
		h->fifo.gp.writeptr = PTR2ADDRESS(h->fifo.pTempWriteptr);
		signalGPThread(h);
		while(!h->fifo.gp.idle) {
		DWORD res;
		GLE(GetExitCodeThread(h->gp->getThread(), &res));
		if(res == GPTR_ERROR)
		break;
		}
		} else*/ if(MASKED_NEQUAL(DWORD(old_pWrite), DWORD(h->fifo.pTempWriteptr), 32)) {
			//we've passed a 32-byte boundary
			h->fifo.gp.writeptr = h->fifo.cpu.writeptr =
				FLOOR_32B(PTR2ADDRESS(h->fifo.pTempWriteptr));
			if(h->fifo.gp.idle) {
				signalGPThread(h);
			}
			/*while(!h->fifo.gp.idle) {	//remove this till Release
			DWORD res;
			GLE(GetExitCodeThread(h->gp->getThread(), &res));
			if(res == GPTR_ERROR)
			break;
			}*/
			//check watermarks here
			DWORD readptr = h->fifo.gp.readptr;
			DWORD dist = CALCULATE_DIST(h->fifo.cpu.writeptr, readptr, h->fifo.cpu.base,
				PAD_ENDPTR(h->fifo.cpu.end));
			//h->fifo.gp.dist = dist;
			if(!h->fifo.overflowed && dist > h->fifo.gp.hiwater) {
				//throw hardware_fatal_exception("GP High watermark reached!");
				FIFODEGUB("GP High watermark reached.\n");
				ESETHWFLAGS(CPSR, CPSR_OVERFLOW);
				if(getbitr(h->hrh(CPCR), 2)) {
					h->interrupt.raise(INTEX_CP, "GP Overflow");
					FIFODEGUB("Overflow interrupt raised.\n");
				}
				h->fifo.overflowed = true;
			}
		}
	} else {  //not linked
		h->fifo.cpu.writeptr = FLOOR_32B(PTR2ADDRESS(h->fifo.pTempWriteptr)); //may be slow
	}
}

void HwGX::do_underflow(Hardware *h) {
	if(h->fifo.overflowed) {
		FIFODEGUB("GP Low watermark reached.\n");
		ESETHWFLAGS(CPSR, CPSR_UNDERFLOW);
		if(getbitr(h->hrh(CPCR), 3)) {
			h->interrupt.raise(INTEX_CP, "GP Underflow");
			FIFODEGUB("Underflow interrupt raised.\n");
		}
		h->fifo.overflowed = false;
	}
}

void HwGX::wrapWrite(Hardware *h, BYTE *data, size_t size) {
	FIFODEGUB("CPU FIFO wrapWrite!\n");
	for(size_t i=0; i<size; i++) {
		*(h->fifo.pTempWriteptr++) = data[i];
		if(h->fifo.pTempWriteptr >= h->fifo.pEnd) {
			h->fifo.pTempWriteptr = h->fifo.pBase;
			h->fifo.cpu.wrap = true;
		}
	}
}

void HwGX::signalGPThread(Hardware *h) {
	FIFODEGUB("Setting GP Event...\n");
	HWGLE(SetEvent(h->gp->getEvent()));
	FIFODEGUB("GP Event set @ cycle %I64u\n", h->m_cc.cycles);
}

void HwGX::wh_pe_isr(Hardware *h, WORD offset, WORD data) {
	//ADDITIVE_TIMING(&h->m_hwt[HWT_GX]);
	MYASSERT(offset == 0x100A);
	WORD orig = h->hrh(offset);
	WORD changed = (WORD)CLEARINTFLAGS(orig, data, 0x000C);
	GPDEGUB("PE_ISR wh: 0x%04X + 0x%04X => 0x%04X\n", orig, data, changed);
	h->hwh(offset, changed);
}

//Fix these functions. use small, unique error-check functions.
//or... collect the error-checks into one point.
//for cpu fifo, HID2 WPE set.
//for gp fifo, read enable set.
//it is illegal to write while the respective bits are set.

void HwGX::ww_cpufifo_base(Hardware *h, WORD offset, DWORD data) {
	MYASSERT(offset == 0x300C);
	GPDEGUB("CPU FIFO %s set: 0x%08X\n", "base", data);
	if(data != h->hrw(offset) && getbitr(h->hrh(0x0002), 4) && getbitr(h->hrh(0x0002), 0))
		throw hardware_fatal_exception("CPU FIFO base change while link enabled!");
	h->fifo.cpu.base = data;
	h->fifo.pBase = h->m.getp_physical(h->fifo.cpu.base,
		PAD_ENDPTR(h->fifo.cpu.end) - h->fifo.cpu.base);
	h->fifo.pEnd = h->fifo.pBase + PAD_ENDPTR(h->fifo.cpu.end) - h->fifo.cpu.base;
	h->hww(offset, data);
}
void HwGX::ww_cpufifo_end(Hardware *h, WORD offset, DWORD data) {
	MYASSERT(offset == 0x3010);
	GPDEGUB("CPU FIFO %s set: 0x%08X\n", "end", data);
	if(data != h->hrw(offset) && getbitr(h->hrh(0x0002), 4) && getbitr(h->hrh(0x0002), 0))
		throw hardware_fatal_exception("CPU FIFO end change while link enabled!");
	h->fifo.cpu.end = data;//MIN(data, MAIN_MEMORY_SIZE-1);
	h->fifo.pBase = h->m.getp_physical(h->fifo.cpu.base,
		PAD_ENDPTR(h->fifo.cpu.end) - h->fifo.cpu.base);
	h->fifo.pEnd = h->fifo.pBase + PAD_ENDPTR(h->fifo.cpu.end) - h->fifo.cpu.base;
	h->hww(offset, data);
}
void HwGX::ww_cpufifo_writeptr(Hardware *h, WORD offset, DWORD data) {
	MYASSERT(offset == 0x3014);
	GPDEGUB("CPU FIFO %s set: 0x%08X\n", "writeptr", data);
	if(getbitr(h->hrh(0x0002), 4) && getbitr(h->hrh(0x0002), 0)) {
		if(data != h->fifo.cpu.writeptr)
			throw hardware_fatal_exception("CPU FIFO writeptr change while link enabled!");
		else if(!fifo_is_flushed(h))
			throw hardware_fatal_exception("CPU FIFO writeptr write while fifo not flushed!");
		else if(!h->fifo.gp.idle)
			throw hardware_fatal_exception("CPU FIFO writeptr write while GP not idle!");
		else if(h->fifo.gp.in_command)
			throw hardware_fatal_exception("CPU FIFO writeptr write while GP in command!");
		h->fifo.gp.writeptr = h->fifo.gp.readptr = data;
		h->fifo.gp.dist = 0;
	}
	h->fifo.cpu.writeptr = data;
	h->fifo.pTempWriteptr = h->m.getp_physical(data, PAD_ENDPTR(h->fifo.cpu.end) - data);
	h->fifo.cpu.wrap = false;
	h->hww(offset, data);
}

DWORD HwGX::rw_cpufifo_base(Hardware *h, WORD offset) {
	MYASSERT(offset == 0x300C);
	DWORD data = h->fifo.cpu.base;
	GPDEGUB("CPU FIFO %s read: 0x%08X\n", "base", data);
	return data;
}
DWORD HwGX::rw_cpufifo_end(Hardware *h, WORD offset) {
	MYASSERT(offset == 0x3010);
	DWORD data = h->fifo.cpu.end;
	GPDEGUB("CPU FIFO %s read: 0x%08X\n", "end", data);
	return data;
}
DWORD HwGX::rw_cpufifo_writeptr(Hardware *h, WORD offset) {
	MYASSERT(offset == 0x3014);
	MYASSERT((h->fifo.cpu.writeptr & 31) == 0);
	DWORD data = h->fifo.cpu.writeptr;
	if(h->fifo.cpu.wrap)
		data |= 0x04000000;
	GPDEGUB("CPU FIFO %s read: 0x%08X\n", "writeptr", data);
	//if(!h->fifo.gp.idle)
	//throw hardware_fatal_exception("CPU FIFO writeptr read while not idle!");
	//if(!fifo_is_flushed(h))
	//throw hardware_fatal_exception("CPU FIFO writeptr read while fifo not flushed!");
	return data;
}

//templatize this
#define RHLO_SIMPLE(word) (word)
#define RHLO_ALIGN FLOOR_32B
#define RHLO_DIST(word) WORD(update_dist(h))

#define RHHI_SIMPLE(word) (word)
#define RHHI_ALIGN(word) (word)
#define RHHI_DIST(word) WORD(update_dist(h) >> 16)

DWORD HwGX::update_dist(Hardware *h) {
	return h->fifo.gp.dist = CALCULATE_DIST(h->fifo.gp.writeptr, h->fifo.gp.readptr,
		h->fifo.gp.base, PAD_ENDPTR(h->fifo.gp.end));
}

#define DEFINE_GXI_GP(cap, gem, address, rhlo)\
	void HwGX::wh_gpfifo_##gem##_lo(Hardware *h, WORD offset, WORD data) {\
	MYASSERT(offset == address);\
	GPDEGUB("Hardware wh: 0x%04X, 0x%04X\n", offset, data);\
	DWORD gem = (h->fifo.gp.gem & ~0xFFFF) | data;\
	GPDEGUB("GP FIFO %s set: 0x%08X\n", #gem, gem); /*}*/\
	if(getbitr(h->hrh(0x0002), 0))\
	throw hardware_fatal_exception("GP FIFO " #gem " write while read enabled!");\
	else if(data != h->hrh(offset))\
	handle_gp_fifo_change(h);\
	h->fifo.gp.gem = gem;\
	h->hwh(offset, data);\
}\
	void HwGX::wh_gpfifo_##gem##_hi(Hardware *h, WORD offset, WORD data) {\
	MYASSERT(offset == address + 2);\
	GPDEGUB("Hardware wh: 0x%04X, 0x%04X\n", offset, data);\
	DWORD gem = (h->fifo.gp.gem & 0xFFFF) | (DWORD(data) << 16);\
	GPDEGUB("GP FIFO %s set: 0x%08X\n", #gem, gem);\
	if(getbitr(h->hrh(0x0002), 0))\
	throw hardware_fatal_exception("GP FIFO " #gem " write while read enabled!");\
	else if(data != h->hrh(offset))\
	handle_gp_fifo_change(h);\
	h->fifo.gp.gem = gem;\
	h->hwh(offset, data);\
}\
	WORD HwGX::rh_gpfifo_##gem##_hi(Hardware *h, WORD offset) {\
	MYASSERT(offset == address + 2);\
	WORD data = RHHI_##rhlo(WORD(h->fifo.gp.gem >> 16));\
	GPDEGUB("GP FIFO %s hi read: 0x%04X\n", #gem, data); return data;\
}\
	WORD HwGX::rh_gpfifo_##gem##_lo(Hardware *h, WORD offset) {\
	MYASSERT(offset == address);\
	WORD data = RHLO_##rhlo(WORD(h->fifo.gp.gem));\
	GPDEGUB("GP FIFO %s lo read: 0x%04X\n", #gem, data); return data;\
}
GXI_GPS(DEFINE_GXI_GP)

void HwGX::handle_gp_fifo_change(Hardware *h) {
	if(!h->fifo.copy_fifo && (!fifo_is_flushed(h) || h->fifo.gp.in_command)) {
		h->fifo.copy_fifo = true;
		h->fifo.copy.base = h->fifo.cpu.base;
		h->fifo.copy.end = h->fifo.cpu.end;
		h->fifo.copy.writeptr = h->fifo.cpu.writeptr;
		if(h->fifo.copy.base != h->fifo.gp.base || h->fifo.copy.end != h->fifo.gp.end ||
			h->fifo.copy.writeptr != h->fifo.gp.writeptr)
			throw hardware_fatal_exception("Unemulated GP FIFO action type 1!");
		h->fifo.copy.pBase = h->fifo.pBase;
		h->fifo.copy.pEnd = h->fifo.pEnd;
		h->fifo.copy.pTempWriteptr = h->fifo.pTempWriteptr;
	}
}

void HwGX::wh_cp_cr(Hardware *h, WORD offset, WORD data) {
	MYASSERT(offset == 0x0002);
	GPDEGUB("CP CR write: 0x%04X", data);
	WORD changed_bits = h->hrh(offset) ^ data;
	const char *names[] = { "FIFO read", "BP irq", "FIFO overflow irq",
		"FIFO underflow irq", "GP-CPU link", "Breakpoint" };
	for(int i=0; i<6; i++) {
		if(getbitr(changed_bits, i)) {
			GPDEGUB(" | %s %s", names[i], abled(getbitr(data, i)));
		}
	}
	GPDEGUB("\n");

	h->fifo.linked = getbitr(data, 4);
	if(getbitr(changed_bits, 4)) {
		if(getbitr(data, 4)) {
			check_fifo_link(h);
		} else {
			stabilize_fifo(h);
		}
	}

	if(getbitr(changed_bits, 0)) {
		if(getbitr(data, 0)) {
			if(h->fifo.copy_fifo) {
				h->fifo.copy_fifo = false;
				if(h->fifo.pTempWriteptr != h->fifo.pBase)
					throw hardware_fatal_exception("Unemulated GP FIFO action type 2!");
				BYTE *floor = (BYTE*)FLOOR_32B(DWORD(h->fifo.copy.pTempWriteptr));
				BYTE *src1 = floor-32 < h->fifo.copy.pBase ? h->fifo.copy.pEnd-32 : floor-32;
				memcpy(h->fifo.pEnd-32, src1, 32);
				DWORD size = h->fifo.copy.pTempWriteptr - floor;
				memcpy(h->fifo.pBase, floor, size);
				h->fifo.pTempWriteptr += size;
			}
			check_fifo_read(h);
			h->gp->fifo_set();
			h->fifo.gp.read_enabled = true;
			signalGPThread(h);
		} else {
			stabilize_fifo(h);
			h->fifo.gp.read_enabled = false;
		}
	}

	if(getbitr(data, 1) || getbitr(data, 5)) {
		if(!getbitr(data, 1))
			throw hardware_fatal_exception("GP breakpoint set badly!");
		if(!getbitr(data, 5)) { //bp irq disable/ack?
			ECLEARHWFLAGS(CPSR, CPSR_BREAKPOINT);
		}
		h->fifo.gp.bp_enabled = true;
	} else {
		h->fifo.gp.bp_enabled = false;
		if(h->fifo.gp.breakpoint_reached) {
			h->fifo.gp.read_enabled = true;
			signalGPThread(h);
		}
	}

	h->hwh(offset, data);
}

void HwGX::stabilize_fifo(Hardware *h) {
	signalGPThread(h);
	Sleep(0);
	if(!h->fifo.gp.idle) {
		FIFODEGUB("Waiting for GP to become idle...\n");
		while(!h->fifo.gp.idle) {
			Sleep(0);
		}
		FIFODEGUB("Wait for GP to become idle complete.\n");
	}

	update_dist(h);
	if(g::fifo_log && h->fifo.linked) {
		DUMPDWORD(h->fifo.cpu.writeptr);
		DUMPDWORD(h->fifo.gp.writeptr);
		DUMPDWORD(h->fifo.gp.readptr);
		DUMPDWORD(h->fifo.cpu.base);
		DUMPDWORD(h->fifo.cpu.end);
		DUMPDWORD(h->fifo.gp.dist);
	}
	if(h->fifo.linked && h->fifo.cpu.writeptr != h->fifo.gp.writeptr)
		throw hardware_fatal_exception("GP FIFO Writeptr discrepancy!");
	/*RELEASE_ASSERT(h->fifo.linked ? (h->fifo.gp.dist ==
	CALCULATE_DIST(h->fifo.cpu.writeptr, h->fifo.gp.readptr, h->fifo.cpu.base,
	PAD_ENDPTR(h->fifo.cpu.end))) : true);*/
	//if(!fifo_is_flushed(h))
	//throw hardware_fatal_exception("GP FIFO read disabled while not flushed!");
	//this may cause trouble, but as long as the fifo is properly flushed,
	//nothing important should get lost.
	/*h->fifo.cpu.writeptr &= ~31;
	h->fifo.gp.writeptr &= ~31;
	h->fifo.gp.readptr &= ~31;*/
	//h->fifo.gp.dist &= ~31;
	update_dist(h);
	if(g::fifo_log && h->fifo.linked) {
		DUMPDWORD(h->fifo.gp.dist);
	}
	//if(h->fifo.linked && h->fifo.cpu.writeptr != h->fifo.gp.writeptr)
	//throw hardware_fatal_exception("GP FIFO Writeptr discrepancy!");
	/*RELEASE_ASSERT(h->fifo.linked ? (h->fifo.gp.dist ==
	CALCULATE_DIST(h->fifo.cpu.writeptr, h->fifo.gp.readptr, h->fifo.cpu.base,
	PAD_ENDPTR(h->fifo.cpu.end))) : true);*/
	//RELEASE_ASSERT(!h->fifo.gp.in_command);
}

bool HwGX::fifo_is_flushed(Hardware *h) {
	static const BYTE zerobuf[32] = {0};
	DWORD size = DWORD(h->fifo.pTempWriteptr) & 31;
	MYASSERT(size < 32);
	return memcmp(zerobuf, (void*)FLOOR_32B(DWORD(h->fifo.pTempWriteptr)), size) == 0;
}

void HwGX::check_fifo_link(Hardware *h) {
	if(h->fifo.cpu.base != h->fifo.gp.base || h->fifo.cpu.end != h->fifo.gp.end)
		throw hardware_fatal_exception("GP-CPU FIFO link with different fifos!");
	if(h->fifo.cpu.writeptr != h->fifo.gp.writeptr) {
		//if(h->fifo.cpu.writeptr == FLOOR_32B(h->fifo.gp.writeptr)) {
		//h->fifo.gp.readptr = h->fifo.gp.writeptr = FLOOR_32B(h->fifo.gp.writeptr);
		//} else
		throw hardware_fatal_exception("GP-CPU FIFO link with writeptrs out of sync!");
	}
	//RELEASE_ASSERT(!h->fifo.gp.in_command);
}
void HwGX::check_fifo_read(Hardware *h) {
	if(!IS_32B_ALIGNED(h->fifo.gp.writeptr))
		throw hardware_fatal_exception("GP FIFO Writeptr unaligned!");
	if(!IS_32B_ALIGNED(h->fifo.gp.readptr))
		throw hardware_fatal_exception("GP FIFO Readptr unaligned!");
	if(!IS_32B_ALIGNED(h->fifo.gp.dist))
		throw hardware_fatal_exception("GP FIFO Dist unaligned!");
	if(WRAP_GPFIFOPTR(h->fifo.gp.readptr + h->fifo.gp.dist, h->fifo.gp) !=
		h->fifo.gp.writeptr)
		throw hardware_fatal_exception("GP FIFO Write/Read/Dist mismatch!");

	//if(!IS_ENDPTR_ALIGNED(h->fifo.gp.end))
	//throw hardware_fatal_exception("GP FIFO bad End pointer!");
	if(!IS_32B_ALIGNED(h->fifo.gp.base))
		throw hardware_fatal_exception("GP FIFO Base unaligned!"); 
	if(h->fifo.gp.end < h->fifo.gp.base)
		throw hardware_fatal_exception("GP FIFO End < Base!");
	if(GPFIFO_SIZE < 64*K)
		throw hardware_fatal_exception("GP FIFO too small!");

	if(!IS_WITHIN(h->fifo.gp.writeptr, h->fifo.gp.base, PAD_ENDPTR(h->fifo.gp.end)))
		throw hardware_fatal_exception("GP FIFO Writeptr out of bounds!");
	if(!IS_WITHIN(h->fifo.gp.readptr, h->fifo.gp.base, PAD_ENDPTR(h->fifo.gp.end)))
		throw hardware_fatal_exception("GP FIFO Readptr out of bounds!");

	if(!IS_32B_ALIGNED(h->fifo.gp.hiwater))
		throw hardware_fatal_exception("GP FIFO Hiwater unaligned!");
	if(!IS_32B_ALIGNED(h->fifo.gp.lowater))
		throw hardware_fatal_exception("GP FIFO Lowater unaligned!");
	if(h->fifo.gp.lowater > h->fifo.gp.hiwater)
		throw hardware_fatal_exception("GP FIFO Lowater bigger than Hiwater!");
	if(h->fifo.gp.lowater >= GPFIFO_SIZE)
		throw hardware_fatal_exception("GP FIFO Lowater bigger than FIFO!");
	if(h->fifo.gp.hiwater >= GPFIFO_SIZE)
		throw hardware_fatal_exception("GP FIFO Hiwater bigger than FIFO!");
	//RELEASE_ASSERT(!h->fifo.gp.in_command);
}
void HwGX::check_wpe(Hardware *h) { //not much use anymore
	if(!IS_32B_ALIGNED(h->fifo.cpu.writeptr))
		throw hardware_fatal_exception("CPU FIFO Writeptr unaligned!");
	if(!IS_32B_ALIGNED(h->fifo.cpu.base))
		throw hardware_fatal_exception("CPU FIFO Base unaligned!");
	//if(!IS_ENDPTR_ALIGNED(h->fifo.cpu.end))
	//throw hardware_fatal_exception("CPU FIFO bad End pointer!");
	if(PAD_ENDPTR(h->fifo.cpu.end) - h->fifo.cpu.base < 64*K)
		throw hardware_fatal_exception("CPU FIFO too small!");
	//RELEASE_ASSERT(!h->fifo.gp.in_command);
}

void HwGX::wh_cp_clear(Hardware *h, WORD offset, WORD data) {
	MYASSERT(offset == 0x0004);
	GPDEGUB("CP Clear write: 0x%04X", data);
	const char *names[] = { "FIFO overflow", "FIFO underflow", "GP Metrics" };
	for(int i=0; i<3; i++) {
		if(getbitr(data, i)) {
			GPDEGUB(" | %s", names[i]);
		}
	}
	GPDEGUB("\n");
	if(data & ~0x0007)
		throw hardware_fatal_exception("GP Clear bad data!");
	h->hwh(offset, data);
	h->hwh(0x0000, h->hrh(0x0000) & (WORD)~(data & 0x0003));
}

void HwGX::wh_cp_metric(Hardware *h, WORD offset, WORD data) {
	MYASSERT(offset == 0x0006);
	switch(data) {
#define PERF_CP_CASE(name, num) case num: DEGUB("GP CP Perf1 set: %s\n", #name); break;
		PERF_CP(PERF_CP_CASE);
	default:
		DEGUB("GP Unknown CP Perf1: 0x%08X\n", data);
		throw hardware_fatal_exception("GP Unknown CP Perf1!");
	};
	h->hwh(offset, data);
}

WORD HwGX::rh_cp_sr(Hardware *h, WORD offset) {
	MYASSERT(offset == 0x0000);
	WORD data = h->hrh(offset);
	/*if(h->fifo.gp.idle && h->fifo.gp.in_command)
	data |= 0x0C;
	else
	data &= ~0x0C;*/
	if(h->fifo.gp.idle)
		data |= 0x04;
	else
		data &= ~0x04;
	if(h->fifo.gp.in_command)
		data |= 0x08;
	else
		data &= ~0x08;
	GPDEGUB("CP Status read: 0x%04X\n", data);
	return data;
}
